package furny.ga.util;

import java.awt.Component;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import furny.entities.Furniture;
import furny.furndb.FurnDBManager;
import furny.ga.FurnEntry;
import furny.ga.FurnEntryList;
import furny.ga.FurnLayoutIndividual;
import furny.ga.RoomVector;

/**
 * Utility class for handling furniture IO.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class FurnLayoutIOUtil {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(FurnLayoutIOUtil.class
      .getName());

  private static File lastDir = new File(".");

  /**
   * Instantiation is not allowed.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private FurnLayoutIOUtil() {
  }

  /**
   * Prints the simple genotype to a string.
   * 
   * @param store
   *          The simple genotype.
   * @return The string of the genotype.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static String printSimpleGenotype(final long[][] store) {
    final StringBuilder sb = new StringBuilder();

    writeSimpleGenotypeHeader(sb);
    writeSimpleGenotype(store, sb);

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine(sb.toString());
    }

    return sb.toString();
  }

  /**
   * Prints the simple genotypes of a list of individuals to a string.
   * 
   * @param list
   *          The list of individuals.
   * @return The string representation.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static String printGenotypes(final List<FurnLayoutIndividual> list) {
    final StringBuilder sb = new StringBuilder();
    for (final FurnLayoutIndividual ind : list) {
      sb.append(ind.getId());
      sb.append(": ");
      writeSimpleGenotype(ind.getSimpleGenotype(), sb);
    }

    if (LOGGER.isLoggable(Level.FINE)) {
      LOGGER.fine(sb.toString());
    }

    return sb.toString();
  }

  /**
   * Creates a furniture layout individual from a string.
   * 
   * @param string
   *          The string to parse.
   * @return The individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static FurnLayoutIndividual parseIndividual(final String string) {
    final FurnLayoutIndividual ind = new FurnLayoutIndividual();
    if (parse(ind, string)) {
      return ind;
    } else {
      throw new RuntimeException("Error parsing individual");
    }
  }

  /**
   * Parses the content of a furniture layout individual from a string.
   * 
   * @param ind
   *          Individual to parse to.
   * @param string
   *          The string to parse.
   * @return The individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static boolean parse(final FurnLayoutIndividual ind,
      final String string) {
    if (ind == null) {
      LOGGER.severe("No Individual given");
      return false;
    }

    final FurnEntryList newGenes = new FurnEntryList();

    final Pattern regex = Pattern.compile("\\{(.*?)\\}", Pattern.DOTALL);
    final Matcher matcher = regex.matcher(string.replaceAll(" ", ""));
    final Pattern regex2 = Pattern
        .compile("\\(([^,]+),([^,]+),([^,]+),([^\\)]+)");
    if (matcher.find()) {
      final String genotype = matcher.group(1);
      final Matcher matcher2 = regex2.matcher(genotype);
      while (matcher2.find()) {
        try {
          final long id = Integer.valueOf(matcher2.group(1));
          final int x = Integer.valueOf(matcher2.group(2));
          final int y = Integer.valueOf(matcher2.group(3));
          final int rotationSteps = Integer.valueOf(matcher2.group(4));

          final Furniture f = FurnDBManager.getInstance().getFurniture(id);

          if (f == null) {
            throw new Exception("No furniture for id " + id);
          }

          final FurnEntry entry = new FurnEntry(new RoomVector(x, y,
              rotationSteps), f);

          newGenes.add(entry);
        } catch (final Exception e) {
          LOGGER.warning(e.toString());
          return false;
        }
      }
    }

    ind.getFurnitures().clear();
    ind.getFurnitures().addAll(newGenes);

    return true;
  }

  /**
   * Writes the header of a simple genotype to a string builder.
   * 
   * @param sb
   *          The string builder.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void writeSimpleGenotypeHeader(final StringBuilder sb) {
    sb.append("FurnID;x;y;rotation\n");
  }

  /**
   * Prints the simple genotype to a string builder.
   * 
   * @param store
   *          Simple genotype.
   * @param sb
   *          String builder.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void writeSimpleGenotype(final long[][] store,
      final StringBuilder sb) {
    for (int i = 0; i < store.length; i++) {
      sb.append(store[i][0]);
      sb.append(';');
      sb.append(store[i][1]);
      sb.append(';');
      sb.append(store[i][2]);
      sb.append(';');
      sb.append(store[i][3]);
      sb.append('\n');
    }
  }

  /**
   * Prints the simple genotype in set notation to a string builder.
   * 
   * @param store
   *          The simple genotype.
   * @param sb
   *          The string builder to write to.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void writeSimpleGenotypeSet(final long[][] store,
      final StringBuilder sb) {
    sb.append('{');
    boolean first = true;

    for (int i = 0; i < store.length; i++) {
      if (!first) {
        sb.append(',');
      }
      first = false;

      sb.append('(');
      sb.append(store[i][0]);
      sb.append(',');
      sb.append(store[i][1]);
      sb.append(',');
      sb.append(store[i][2]);
      sb.append(',');
      sb.append(store[i][3]);
      sb.append(')');
    }
    sb.append('}');
  }

  /**
   * Writes the fitness of a furniture layout individual to a string builder.
   * 
   * @param ind
   *          The individual.
   * @param sb
   *          The string builder to write to.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static void writeFitness(final FurnLayoutIndividual ind,
      final StringBuilder sb) {
    sb.append("Fitness: ");
    if (ind.isEvaluated()) {
      sb.append(ind.getFitness());
    } else {
      sb.append("unevaluated");
    }
    sb.append('\n');
  }

  /**
   * Loads a furniture layout individual by opening a {@link JFileChooser},
   * loading and parsing the file.
   * 
   * @param parent
   *          Parent for the file chooser, may be <code>null</code>.
   * @param ind
   *          The individual to write the genotype to, may be <code>null</code>.
   * @return The loaded individual.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static FurnLayoutIndividual loadGenotype(final Component parent,
      final FurnLayoutIndividual ind) {
    final FurnLayoutIndividual fli = ind != null ? ind
        : new FurnLayoutIndividual();

    final JFileChooser fc = new JFileChooser(lastDir);
    fc.setFileFilter(new ExtensionFileFilter("Furny individuals (*.fi)", "fi"));
    final int n = fc.showOpenDialog(parent);

    if (n == JFileChooser.APPROVE_OPTION) {
      final File f = fc.getSelectedFile();

      if (f != null) {
        lastDir = fc.getCurrentDirectory();

        BufferedReader fr = null;
        try {
          fr = new BufferedReader(new FileReader(f));

          final String content = fr.readLine();

          if (FurnLayoutIOUtil.parse(fli, content)) {
            return fli;
          }

        } catch (final Exception e1) {
          JOptionPane.showMessageDialog(parent, e1.toString(),
              "Error writing file", JOptionPane.WARNING_MESSAGE);
          LOGGER.log(java.util.logging.Level.WARNING, "Error loading file", e1);
        } finally {
          try {
            fr.close();
          } catch (final Exception e2) {
          }
        }
      }
    }
    return null;
  }

  /**
   * Saves a furniture layout individual by opening a {@link JFileChooser},
   * writing and saving the file.
   * 
   * @param parent
   *          Parent for the file chooser, may be <code>null</code>.
   * @param ind
   *          The individual to save.
   * @return The status, {@link JOptionPane#OK_OPTION} if all is ok.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public static int saveGenotype(final Component parent,
      final FurnLayoutIndividual ind) {
    boolean ok = true;

    do {
      final JFileChooser fc = new JFileChooser(lastDir);
      fc.setFileFilter(new ExtensionFileFilter("Furny individuals (*.fi)", "fi"));
      final int n = fc.showSaveDialog(parent);

      if (n == JFileChooser.APPROVE_OPTION) {
        File f = fc.getSelectedFile();

        if (f != null) {
          lastDir = fc.getCurrentDirectory();

          if (!f.getName().endsWith(".fi")) {
            f = new File(f.getAbsolutePath() + ".fi");
          }

          if (f.exists()) {
            final int ync = JOptionPane.showConfirmDialog(parent,
                "File already exists. Do you want to overwrite?",
                "File exists", JOptionPane.YES_NO_CANCEL_OPTION);
            switch (ync) {
            case JOptionPane.YES_OPTION:
              ok = true;
              break;

            case JOptionPane.NO_OPTION:
              ok = false;
              break;

            case JOptionPane.CANCEL_OPTION:
            default:
              return JOptionPane.CANCEL_OPTION;
            }
          }

          if (ok) {
            BufferedWriter fw = null;
            try {
              fw = new BufferedWriter(new FileWriter(f));

              fw.write(ind.getGenotypeString());

              JOptionPane.showMessageDialog(parent,
                  "Individual successfully saved", "Success",
                  JOptionPane.INFORMATION_MESSAGE);
            } catch (final Exception e1) {
              JOptionPane.showMessageDialog(parent, e1.toString(),
                  "Error writing file", JOptionPane.WARNING_MESSAGE);
              LOGGER.log(java.util.logging.Level.WARNING, "Error writing file",
                  e1);
              ok = false;
            } finally {
              try {
                fw.close();
              } catch (final Exception e2) {
              }
            }
          }
        }
      } else {
        return JOptionPane.CANCEL_OPTION;
      }
    } while (!ok);

    return JOptionPane.OK_OPTION;
  }
}
